package Shader;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.lwjgl.BufferUtils;
import org.lwjgl.opengl.GL11;
import org.lwjgl.opengl.GL13;
import org.lwjgl.opengl.GL20;
import org.lwjgl.util.vector.Matrix4f;
import org.lwjgl.util.vector.Vector3f;

import Camera.Camera;
import Model.GUI;
import Model.Model;

public class Shader {
	private int projectionUniformMatrixId;
	private int viewUniformMatrixId;
	public int transformationUniformMatrixId;
	private FloatBuffer matrixBuffer = BufferUtils.createFloatBuffer(16);
	
	public Map<String, Integer> uniformIds = new HashMap<String, Integer>();

	private int programId;
	private List<Model> models = new ArrayList<>();
	private List<GUI> gui = new ArrayList<>();
	public static List<Shader> shaders = new ArrayList<>();
	
	public boolean ui;
	public boolean visible = true;
	
    private void push(Shader push) {
    	shaders.add(push);
    }
    public void pushModel(Model push) {
    	models.add(push);
    }
    public void pushGUI(GUI push) {
    	gui.add(push);
    }
	public static Vector3f Color(int hex) {
		var r = (float) (hex >> 16 & 255) / 255f;
		var g = (float) (hex >> 8 & 255) / 255f;
		var b = (float) (hex & 255) / 255f;

		return new Vector3f(r, g, b);
	}
	
	private int createShader(int type, String path) {
        StringBuilder source = new StringBuilder();
        try {
            BufferedReader reader = new BufferedReader(new FileReader(path));
            String line;
            while((line = reader.readLine()) != null) {
            	source.append(line).append("//\n");
            }
            reader.close();
        } catch(IOException e) {
            e.printStackTrace();
            System.exit(-1);
        }
        
		int shader = GL20.glCreateShader(type);

	    GL20.glShaderSource(shader, source);
	    GL20.glCompileShader(shader);
	    
        if(GL20.glGetShaderi(shader, GL20.GL_COMPILE_STATUS ) == GL11.GL_FALSE){
            System.out.println(GL20.glGetShaderInfoLog(shader, 500));
            System.err.println("Could not compile shader!");
            GL20.glDeleteShader(shader);
            System.exit(-1);
        }
        
        return shader;
	}
	private int createProgram(int vertexShader, int fragmentShader) {
	    int program = GL20.glCreateProgram();

	    GL20.glAttachShader(program, vertexShader);
	    GL20.glAttachShader(program, fragmentShader);
	    GL20.glLinkProgram(program);
	    
        if(GL20.glGetProgrami(program, GL20.GL_COMPILE_STATUS ) == GL11.GL_FALSE){
            System.out.println(GL20.glGetProgramInfoLog(program, 500));
            System.err.println("Could not compile shader!");
            GL20.glDeleteProgram(program);
            System.exit(-1);
        }
        return program;
	}
	public Shader(String vertexShaderPath, String fragmentShaderPath, String[] uniformNames, boolean ui) {
		int vertexShader = createShader(GL20.GL_VERTEX_SHADER, vertexShaderPath);
		int fragmentShader = createShader(GL20.GL_FRAGMENT_SHADER, fragmentShaderPath);
		
		programId = createProgram(vertexShader, fragmentShader);
		
		this.ui = ui;
		
		this.push(this);

		GL20.glUseProgram(programId);
		
		for(int i = 0; i < uniformNames.length; i++) {
			uniformIds.put(uniformNames[i], GL20.glGetUniformLocation(programId, uniformNames[i]));
		}
		
		viewUniformMatrixId = GL20.glGetUniformLocation(programId, "viewMatrix");
		projectionUniformMatrixId = GL20.glGetUniformLocation(programId, "projectionMatrix");
		transformationUniformMatrixId = GL20.glGetUniformLocation(programId, "transformationMatrix");
	}
    public void loadMatrix(int location, Matrix4f matrix){
        matrix.store(matrixBuffer);
        matrixBuffer.flip();
        GL20.glUniformMatrix4(location, false, matrixBuffer);
    }
    public void loadFloat(String name, float value){
    	int location = uniformIds.get(name);
        GL20.glUniform1f(location, value);
    }
    public void loadVector(String name, Vector3f vector){
    	int location = uniformIds.get(name);
        GL20.glUniform3f(location, vector.x, vector.y, vector.z);
    }
    public void loadBoolean(String name, boolean value){
    	int location = uniformIds.get(name);
        float toLoad = 0;
        if(value){
            toLoad = 1;
        }
        GL20.glUniform1f(location, toLoad);
    }
    public int getProgram() {
    	return programId;
    }
    public void init(Camera camera) {
    	loadMatrix(projectionUniformMatrixId, camera.projectionMatrix);
    	loadMatrix(viewUniformMatrixId, camera.viewMatrix);
    }
    public void render(Camera camera) {
    	GL20.glUseProgram(programId);
    	init(camera);
    	for(int i = 0; i < models.size(); i++) {
    		if(models.get(i).visible) {
        		models.get(i).update();
        		models.get(i).render();
    		}
    	}
    }
    public void guiRender() {
    	GL20.glUseProgram(programId);
    	for(int i = 0; i < gui.size(); i++) {
    		gui.get(i).render();
    	}
    }
}
